#ifndef F_CPU
#define F_CPU 16000000UL
#endif

#include <avr/io.h>
#include <util/delay.h>
#include "si5351a_optimized.h"
#include "i2c.h"

// Конфигурационные константы
#ifndef CRYSTAL_FREQ
#define CRYSTAL_FREQ 27000000UL //Частота опоры si5351
#endif

#ifndef IF_FREQ
#define IF_FREQ 45000000UL //Частота ПЧ
#endif

#ifndef MAX_DIVIDER
#define MAX_DIVIDER 900UL
#endif

#ifndef MIN_DIVIDER
#define MIN_DIVIDER 8UL
#endif

#ifndef MAX_MULTIPLIER
#define MAX_MULTIPLIER 36
#endif

#ifndef MIN_MULTIPLIER  
#define MIN_MULTIPLIER 15
#endif

// Максимальный знаменатель для Si5351
#define SI5351_MAX_DENOM 1048575UL

// Строгий допуск для целочисленного режима (5Гц)
#define INTEGER_TOLERANCE_STRICT 5

// Функция упрощения дроби (алгоритм Евклида)
static void simplify_fraction(uint32_t *num, uint32_t *denom) {
    if (*num == 0 || *denom == 0) {
        return;
    }
    
    uint32_t a = *num;
    uint32_t b = *denom;
    uint32_t temp;
    
    // Алгоритм Евклида для нахождения НОД
    while (b != 0) {
        temp = b;
        b = a % b;
        a = temp;
    }
    
    uint32_t gcd = a;
    if (gcd > 1) {
        *num /= gcd;
        *denom /= gcd;
    }
}

// Поиск наилучшего дробного приближения (без float)
static void find_best_fraction(uint32_t target_freq, uint32_t pll_freq, uint32_t divider, 
                              uint32_t *num, uint32_t *denom) {
    uint32_t actual_freq = pll_freq / divider;
    
    // Вычисляем ошибку частоты
    int32_t error = (int32_t)(target_freq - actual_freq);
    
    *denom = SI5351_MAX_DENOM;
    
    if (error == 0) {
        *num = 0;
        return;
    }
    
    // Расчет числителя для компенсации ошибки (без float)
    // num = |error| * divider * denom / pll_freq
    uint64_t abs_error = (error < 0) ? (uint64_t)(-error) : (uint64_t)error;
    uint64_t numerator = abs_error * divider * (*denom);
    
    // Деление с округлением
    *num = (uint32_t)((numerator + pll_freq / 2) / pll_freq);
    
    // Ограничение числителя
    if (*num > *denom) {
        *num = *denom;
    }
    
    // Корректировка знака
    if (error < 0) {
        *num = *denom - *num;
    }
    
    // Упрощение дроби для минимизации шума
    simplify_fraction(num, denom);
}

// АКТИВНЫЙ поиск целочисленного режима
uint8_t find_integer_mode_solution(uint32_t target_freq, uint8_t *best_mult, uint32_t *best_divider) {
    // Приоритетные множители для минимального фазового шума
    uint8_t optimal_mults[] = {30, 28, 32, 34, 36, 26, 24};
    uint32_t min_error = UINT32_MAX;
    uint8_t found_solution = 0;
    
    for(uint8_t m = 0; m < 7; m++) {
        uint32_t pll_freq = optimal_mults[m] * CRYSTAL_FREQ;
        uint32_t base_divider = pll_freq / target_freq;
        
        // Искать в диапазоне ±15 делителей для максимального покрытия
        for(uint32_t div = base_divider - 15; div <= base_divider + 15; div++) {
            if(div < MIN_DIVIDER || div > MAX_DIVIDER) continue;
            
            uint32_t actual_freq = pll_freq / div;
            uint32_t error = (actual_freq > target_freq) ? 
                           (actual_freq - target_freq) : (target_freq - actual_freq);
            
            // Ищем решение в пределах строгого допуска
            if(error <= INTEGER_TOLERANCE_STRICT && error < min_error) {
                min_error = error;
                *best_mult = optimal_mults[m];
                *best_divider = div;
                found_solution = 1;
                
                // Если нашли идеальное решение (0 Гц ошибка) - сразу возвращаем
                if(error == 0) {
                    return 1;
                }
            }
        }
    }
    
    return found_solution;
}

// Оптимизированная установка PLL (без float)
static void setupPLL_optimized(uint8_t pll, uint8_t mult, uint32_t num, uint32_t denom, uint8_t integer_mode) {
    uint32_t P1, P2, P3;
    
    if (integer_mode || num == 0) {
        // Целочисленный режим - лучший фазовый шум
        P1 = 128UL * mult - 512;
        P2 = 0;
        P3 = 1;
    } else {
        // Дробный режим - БЕЗ FLOAT!
        // P1 = 128 * mult + (128 * num) / denom - 512
        P1 = (128UL * num) / denom;
        P1 = 128UL * mult + P1 - 512;
        
        // P2 = 128 * num - denom * ((128 * num) / denom)
        P2 = (128UL * num) - denom * ((128UL * num) / denom);
        P3 = denom;
    }
    
    // Пакетная запись оптимизированных регистров
    uint8_t pll_regs[8] = {
        (uint8_t)((P3 & 0x0000FF00) >> 8),
        (uint8_t)(P3 & 0x000000FF),
        (uint8_t)((P1 & 0x00030000) >> 16),
        (uint8_t)((P1 & 0x0000FF00) >> 8),
        (uint8_t)(P1 & 0x000000FF),
        (uint8_t)(((P3 & 0x000F0000) >> 12) | ((P2 & 0x000F0000) >> 16)),
        (uint8_t)((P2 & 0x0000FF00) >> 8),
        (uint8_t)(P2 & 0x000000FF)
    };
    
    si5351a_burst_write(pll, pll_regs, 8);
}

// Оптимизированная установка MultiSynth
static void setupMultisynth_optimized(uint8_t synth, uint32_t divider, uint8_t rDiv) {
    uint32_t P1, P2, P3;
    
    P1 = 128UL * divider - 512;
    P2 = 0;
    P3 = 1;
    
    uint8_t ms_regs[8] = {
        (uint8_t)((P3 & 0x0000FF00) >> 8),
        (uint8_t)(P3 & 0x000000FF),
        (uint8_t)(((P1 & 0x00030000) >> 16) | rDiv),
        (uint8_t)((P1 & 0x0000FF00) >> 8),
        (uint8_t)(P1 & 0x000000FF),
        (uint8_t)(((P3 & 0x000F0000) >> 12) | ((P2 & 0x000F0000) >> 16)),
        (uint8_t)((P2 & 0x0000FF00) >> 8),
        (uint8_t)(P2 & 0x000000FF)
    };
    
    si5351a_burst_write(synth, ms_regs, 8);
}

// Дополнительная оптимизация для целочисленного режима
void optimize_for_integer_mode(void) {
    // В целочисленном режиме можно дополнительно снизить шум
    
    // Уменьшить ток фазового детектора (меньше шума)
    i2cSendRegister(SI5351_ADDRESS, 0x8A, 0x50); // Пониженный ток PD
    
    // Включить режим минимального шума
    i2cSendRegister(SI5351_ADDRESS, 0xB0, 0x40); // Low noise mode
    
    // Увеличить подавление питания PLL
    i2cSendRegister(SI5351_ADDRESS, 0x16, 0x1F); // Максимальная фильтрация PLL A
    i2cSendRegister(SI5351_ADDRESS, 0x17, 0x1F); // Максимальная фильтрация PLL B
}

// Конфигурация выхода для минимального шума (фиксированный ток 2mA)
static void configure_output_for_low_phase_noise(void) {
    // Фиксированный ток выхода 2mA для минимального шума
    // 0x4C = 0b01001100 - 2mA, PLL A, MS0 как источник, минимальный шум
    i2cSendRegister(SI5351_ADDRESS, SI_CLK0_CONTROL, 0x4C);
    
    // Отключить spread spectrum
    i2cSendRegister(SI5351_ADDRESS, 149, 0x00);
    
    // Включить буферизацию выхода
    i2cSendRegister(SI5351_ADDRESS, 3, 0xFF); // CLKx control
}

// Выбор оптимального множителя PLL для частоты
static uint8_t select_optimal_multiplier(uint32_t freq) {
    if (freq <= 90000000UL) {
        return 30; // Оптимально для фазового шума до 90 MHz
    } else if (freq <= 112500000UL) {
        return 34; // Для 90-112.5 MHz
    } else if (freq <= 135000000UL) {
        return 36; // Для 112.5-135 MHz (максимум)
    } else {
        return 28; // Безопасный выбор для высоких частот
    }
}

// Корректировка делителя и множителя для оптимального диапазона
static void adjust_divider_and_multiplier(uint32_t freq, uint8_t *mult, uint32_t *divider) {
    // Корректировка слишком маленького делителя
    if (*divider < MIN_DIVIDER) {
        *divider = MIN_DIVIDER;
        *mult = (freq * (*divider) + CRYSTAL_FREQ - 1) / CRYSTAL_FREQ;
        if (*mult > MAX_MULTIPLIER) *mult = MAX_MULTIPLIER;
        if (*mult < MIN_MULTIPLIER) *mult = MIN_MULTIPLIER;
    }
    
    // Корректировка слишком большого делителя
    if (*divider > MAX_DIVIDER) {
        *divider = MAX_DIVIDER;
        *mult = (freq * (*divider) + CRYSTAL_FREQ - 1) / CRYSTAL_FREQ;
        if (*mult > MAX_MULTIPLIER) *mult = MAX_MULTIPLIER;
        if (*mult < MIN_MULTIPLIER) *mult = MIN_MULTIPLIER;
    }
}

// ОСНОВНАЯ ОПТИМИЗИРОВАННАЯ ФУНКЦИЯ УСТАНОВКИ ЧАСТОТЫ
void si5351aSetFrequencyOptimized(uint32_t base_freq) {
    uint32_t z = base_freq + IF_FREQ; // Добавляем ПЧ
    uint32_t pllFreq;
    uint8_t mult;
    uint32_t divider;
    uint32_t num = 0, denom = 1;
    uint8_t use_integer_mode = 0;
    
    // Проверка входных параметров
    if (base_freq == 0) {
        return; // ERROR: нулевая частота
    }
    
    // 1. АКТИВНЫЙ поиск целочисленного режима
    if(find_integer_mode_solution(z, &mult, &divider)) {
        use_integer_mode = 1;
        num = 0;
        denom = 1;
        
        // Дополнительная оптимизация для целочисленного режима
        optimize_for_integer_mode();
    } else {
        // 2. Если целочисленный режим не найден - используем адаптивный алгоритм
        use_integer_mode = 0;
        mult = select_optimal_multiplier(z);
        
        // Расчет делителя
        pllFreq = mult * CRYSTAL_FREQ;
        divider = pllFreq / z;
        
        // Корректировка делителя и множителя
        adjust_divider_and_multiplier(z, &mult, &divider);
        
        // Пересчет PLL частоты после возможной корректировки
        pllFreq = mult * CRYSTAL_FREQ;
        
        // Расчет дробной части
        find_best_fraction(z, pllFreq, divider, &num, &denom);
    }
    
    // Установка PLL и MultiSynth
    setupPLL_optimized(SI_SYNTH_PLL_A, mult, num, denom, use_integer_mode);
    setupMultisynth_optimized(SI_SYNTH_MS_0, divider, SI_R_DIV_1);
    
    // Настройка выхода для минимального шума
    configure_output_for_low_phase_noise();
}

// Функции инициализации и утилиты
void si5351a_fast_init(void) {
    _delay_ms(10);
    
    // Отключить все выходы
    i2cSendRegister(SI5351_ADDRESS, SI_CLK0_CONTROL, 0x80);
    i2cSendRegister(SI5351_ADDRESS, SI_CLK1_CONTROL, 0x80);
    i2cSendRegister(SI5351_ADDRESS, SI_CLK2_CONTROL, 0x80);
    
    // Базовая конфигурация
    i2cSendRegister(SI5351_ADDRESS, 3, 0xFF); // CLKx control
    i2cSendRegister(SI5351_ADDRESS, 149, 0x00); // Отключить spread spectrum
    
    _delay_ms(10);
}

void si5351a_set_frequency_fast(uint32_t freq) {
    si5351aSetFrequencyOptimized(freq);
}

void si5351a_burst_write(uint8_t start_reg, const uint8_t *data, uint8_t length) {
    // Простая проверка параметров
    if (length == 0) {
        return;
    }
    
    i2cStart(SI5351_ADDRESS);
    i2cWrite(start_reg);
    for(uint8_t i = 0; i < length; i++) {
        i2cWrite(data[i]);
    }
    i2cStop();
}

// Оригинальная функция для обратной совместимости
void si5351aSetFrequency(uint32_t z) {
    si5351aSetFrequencyOptimized(z);
}