#ifndef F_CPU
#define F_CPU 16000000UL
#endif

#include <avr/io.h>
#include <util/delay.h>
#include "i2c.h"

#define TWBR_VALUE (((F_CPU / 400000UL) - 16) / 2)

void i2cInit(void) {
    // Настройка скорости I2C ~400 kHz
    TWSR = 0x00; // Prescaler = 1
    TWBR = TWBR_VALUE; // Baud rate
    TWCR = (1 << TWEN); // Enable TWI
}


uint8_t i2cStart(uint8_t address) {
    // Send START condition
    TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
    
    // Wait for TWINT flag set
    while (!(TWCR & (1 << TWINT)));
    
    // Check if START condition was transmitted
    if ((TWSR & 0xF8) != TW_START) {
        return I2C_ERROR;
    }
    
    // Send device address
    TWDR = address & 0xFE; // Write mode
    TWCR = (1 << TWINT) | (1 << TWEN);
    
    while (!(TWCR & (1 << TWINT)));
    
    // Check if device acknowledged
    if ((TWSR & 0xF8) != TW_MT_SLA_ACK) {
        return I2C_ERROR_NACK;
    }
    
    return I2C_SUCCESS;
}

uint8_t i2cRepeatedStart(uint8_t address) {
    // Send REPEATED START condition
    TWCR = (1 << TWINT) | (1 << TWSTA) | (1 << TWEN);
    
    // Wait for TWINT flag set
    while (!(TWCR & (1 << TWINT)));
    
    // Check if REPEATED START condition was transmitted
    if ((TWSR & 0xF8) != TW_REP_START) {
        return I2C_ERROR;
    }
    
    // Send device address
    TWDR = address;
    TWCR = (1 << TWINT) | (1 << TWEN);
    
    while (!(TWCR & (1 << TWINT)));
    
    // Check if device acknowledged
    if (address & 0x01) { // Read mode
        if ((TWSR & 0xF8) != TW_MR_SLA_ACK) {
            return I2C_ERROR_NACK;
        }
    } else { // Write mode
        if ((TWSR & 0xF8) != TW_MT_SLA_ACK) {
            return I2C_ERROR_NACK;
        }
    }
    
    return I2C_SUCCESS;
}

void i2cStop(void) {
    // Send STOP condition
    TWCR = (1 << TWINT) | (1 << TWSTO) | (1 << TWEN);
    
    // Wait for STOP condition to be executed
    _delay_us(10);
}

uint8_t i2cWrite(uint8_t data) {
    TWDR = data;
    TWCR = (1 << TWINT) | (1 << TWEN);
    
    while (!(TWCR & (1 << TWINT)));
    
    if ((TWSR & 0xF8) != TW_MT_DATA_ACK) {
        return I2C_ERROR_NACK;
    }
    
    return I2C_SUCCESS;
}

uint8_t i2cReadAck(void) {
    TWCR = (1 << TWINT) | (1 << TWEN) | (1 << TWEA);
    
    while (!(TWCR & (1 << TWINT)));
    
    return TWDR;
}

uint8_t i2cReadNack(void) {
    TWCR = (1 << TWINT) | (1 << TWEN);
    
    while (!(TWCR & (1 << TWINT)));
    
    return TWDR;
}

uint8_t i2cSendRegister(uint8_t devAddr, uint8_t reg, uint8_t data) {
    uint8_t status = i2cStart(devAddr);
    if (status != I2C_SUCCESS) return status;
    
    status = i2cWrite(reg);
    if (status != I2C_SUCCESS) {
        i2cStop();
        return status;
    }
    
    status = i2cWrite(data);
    i2cStop();
    return status;
}

uint8_t i2cReadRegister(uint8_t devAddr, uint8_t reg, uint8_t *data) {
    uint8_t status = i2cStart(devAddr);
    if (status != I2C_SUCCESS) return status;
    
    status = i2cWrite(reg);
    if (status != I2C_SUCCESS) {
        i2cStop();
        return status;
    }
    
    status = i2cRepeatedStart(devAddr | 0x01);
    if (status != I2C_SUCCESS) return status;
    
    *data = i2cReadNack();
    i2cStop();
    return I2C_SUCCESS;
}

// Совместимость со старым кодом
uint8_t i2cByteSend(uint8_t data) {
    return i2cWrite(data);
}