
#include "oled_ssd1306.h"
#include <string.h>
#include <avr/pgmspace.h>
#include <stdio.h>

static uint8_t oled_buffer[512] = {0};

void oled_send_command(uint8_t cmd) {
    i2cStart(OLED_ADDRESS);
    i2cWrite(0x00);
    i2cWrite(cmd);
    i2cStop();
}

void oled_send_data(uint8_t data) {
    i2cStart(OLED_ADDRESS);
    i2cWrite(0x40);
    i2cWrite(data);
    i2cStop();
}

void oled_init(void) {
    _delay_ms(100);
    
    oled_send_command(0xAE); // Display OFF
    
    oled_send_command(0x20); oled_send_command(0x00); // Horizontal addressing mode
    
    oled_send_command(0x21); oled_send_command(0x00); oled_send_command(0x7F); // Column address
    oled_send_command(0x22); oled_send_command(0x00); oled_send_command(0x03); // Page address
    
    oled_send_command(0x40); // Start line
    oled_send_command(0xA1); // Segment remap
    oled_send_command(0xC8); // COM output scan direction
    oled_send_command(0x81); oled_send_command(0x7F); // Contrast
    oled_send_command(0xA6); // Normal display
    oled_send_command(0xA8); oled_send_command(0x1F); // Multiplex ratio
    oled_send_command(0xD3); oled_send_command(0x00); // Display offset
    oled_send_command(0xD5); oled_send_command(0x80); // Clock divide
    oled_send_command(0xD9); oled_send_command(0x22); // Pre-charge period
    oled_send_command(0xDA); oled_send_command(0x02); // COM pins
    oled_send_command(0xDB); oled_send_command(0x20); // VCOMH deselect level
    oled_send_command(0x8D); oled_send_command(0x14); // Charge pump enable
    
    oled_send_command(0xAF); // Display ON
    
    oled_clear();
    oled_update();
    _delay_ms(100);
}

void oled_clear(void) {
    for(uint16_t i = 0; i < 512; i++) {
        oled_buffer[i] = 0x00;
    }
}

void oled_update(void) {
    for(uint8_t page = 0; page < 4; page++) {
        oled_send_command(0xB0 + page);
        oled_send_command(0x00);
        oled_send_command(0x10);
        
        i2cStart(OLED_ADDRESS);
        i2cWrite(0x40);
        for(uint8_t col = 0; col < 128; col++) {
            i2cWrite(oled_buffer[page * 128 + col]);
        }
        i2cStop();
    }
}

// Основная функция для записи символа с любым шрифтом
void oled_write_char(uint8_t x, uint8_t y, char c, const FontDef *font) {
    if(c < font->first_char || c > font->last_char) {
        c = ' '; // Заменяем непечатные символы на пробел
    }
    
    const uint8_t *char_data = font_get_char_data(font, c);
    
    // Для шрифтов высотой больше 8 пикселей используем несколько строк
    uint8_t rows_needed = (font->height + 7) / 8; // Округляем вверх до целых строк
    
    for(uint8_t row = 0; row < rows_needed; row++) {
        for(uint8_t col = 0; col < font->width; col++) {
            uint8_t data = pgm_read_byte(char_data + row * font->width + col);
            uint8_t target_y = y + row;
            
            if((x + col) < OLED_WIDTH && target_y < 4) {
                oled_buffer[target_y * OLED_WIDTH + (x + col)] = data;
            }
        }
    }
}

// Основная функция для записи строки с любым шрифтом
void oled_write_string(uint8_t x, uint8_t y, const char *str, const FontDef *font) {
    while(*str) {
        oled_write_char(x, y, *str, font);
        x += font->width + 1; // Ширина символа + промежуток
        if(x >= OLED_WIDTH) break;
        str++;
    }
}

// Функции для обратной совместимости
void oled_write_char_small(uint8_t x, uint8_t y, char c) {
    oled_write_char(x, y, c, &font_5x7);
}

void oled_write_string_small(uint8_t x, uint8_t y, const char *str) {
    oled_write_string(x, y, str, &font_5x7);
}

void oled_write_char_large(uint8_t x, uint8_t y, char c) {
    oled_write_char(x, y, c, &font_8x16);
}

void oled_write_string_large(uint8_t x, uint8_t y, const char *str) {
    oled_write_string(x, y, str, &font_8x16);
}

void oled_clear_line(uint8_t y) {
    for(uint8_t col = 0; col < OLED_WIDTH; col++) {
        if(y < 4) {
            oled_buffer[y * OLED_WIDTH + col] = 0x00;
        }
    }
}

// Старая функция для обратной совместимости
void oled_clear_line_small(uint8_t y) {
    oled_clear_line(y);
}

void oled_draw_frequency(uint32_t freq) {
    char buffer[20];
    
    if (freq >= 10000000) {
        sprintf(buffer, "%02lu.%03lu.%03lu", 
                freq / 1000000, 
                (freq % 1000000) / 1000, 
                freq % 1000);
    } else if (freq >= 1000000) {
        sprintf(buffer, "%lu.%03lu.%03lu", 
                freq / 1000000, 
                (freq % 1000000) / 1000, 
                freq % 1000);
    } else {
        sprintf(buffer, "%03lu.%03lu", 
                freq / 1000, 
                freq % 1000);
    }
    
    // Используем большой шрифт для частоты
    uint8_t len = strlen(buffer);
    uint8_t char_width = font_8x16.width + 1;
    uint8_t start_x = (OLED_WIDTH - len * char_width) / 2;
    oled_write_string_large(start_x, 1, buffer);
}
