//Проект синтезатора частоты RN4NU
#ifndef F_CPU
#define F_CPU 16000000UL //16МГц еще лучше
#endif

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <util/atomic.h>
#include <stdio.h>
#include "i2c.h"
#include "si5351a_optimized.h"
#include "oled_ssd1306.h"
#include "fonts.h"
#include "main.h"

// ==================== ГЛОБАЛЬНЫЕ ПЕРЕМЕННЫЕ ====================

// Диапазоны частот,массив
uint32_t freq_max[8] = {29700000, 29000000, 21450000, 14350000, 10150000, 7200000, 3800000, 2000000}; 
uint32_t freq_min[8] = {1810000, 28000000, 21000000, 14000000, 10100000, 7000000, 3500000, 1810000};
uint32_t frequency_1[8] = {1810000, 28050000, 21050000, 14060000, 10120000, 7030000, 3550000, 1810000};

// Названия диапазонов для дисплея
const char* band_names[8] = {"All bnd", "10m", "15m", "20m", "30m", "40m", "80m", "160m"};

// S-meter пороги (примерные значения, нужно калибровать под конкретную аппаратуру)
const uint16_t smeter_thresholds[SMETER_THRESHOLDS_COUNT] = {
    50,   // S1
    100,  // S2
    150,  // S3
    200,  // S4
    250,  // S5
    300,  // S6
    350,  // S7
    400,  // S8
    450,  // S9
    500,  // S9+10
    600   // S9+40
};

// S-meter текстовые значения
const char* smeter_strings[SMETER_THRESHOLDS_COUNT] = {
    "S1", "S2", "S3", "S4", "S5", "S6", "S7", "S8", "S9", "S9+10", "S9+40"
};

// Основные переменные трансивера
volatile int8_t encoderCount = 0;
volatile uint32_t frequency = 1810000;
volatile uint32_t currentFrequency = 1810000;
volatile uint32_t stepSize = 5;     //Начальный шаг перестройки
volatile int32_t ritOffset = 0;
volatile uint8_t ritEnabled = 0;
volatile uint8_t pttPressed = 0;
volatile uint8_t currentBand = 0;    //Стартовый диапазон (первый столбик в массиве)

// S-meter переменные
volatile uint16_t smeterValue = 0;
volatile uint8_t smeterUpdateCounter = 0;

// Переменные для энкодера
static uint8_t oldAB = 0;
const int8_t transitions[] = {0, -1, 1, 0, 1, 0, 0, -1, -1, 0, 0, 1, 0, 1, -1, 0};

// Переменные для антидребезга кнопок
static uint16_t buttonDebounceCounters[BUTTON_COUNT] = {0};
static button_t lastButtonState[BUTTON_COUNT] = {BUTTON_NONE};

// Текущий активный шрифт для разных частей дисплея
const FontDef* status_font = &font_5x7;    // Шрифт для статуса (RX/TX/RIT)
const FontDef* freq_font = &font_8x16;     // Шрифт для частоты (большой)
const FontDef* info_font = &font_5x7;      // Шрифт для информации (маленький)

// ==================== ПРЕРЫВАНИЯ ====================

ISR(PCINT2_vect) {
    uint8_t newAB = (PIND & ((1 << ENCODER_PIN1) | (1 << ENCODER_PIN2))) >> 2;
    
    oldAB <<= 2;
    oldAB |= newAB;
    
    int8_t change = transitions[oldAB & 0x0F];
    
    if (change != 0) {
        ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
            encoderCount += change;
        }
    }
}

// ==================== ФУНКЦИИ ДЛЯ 74HC595 ====================

void shiftRegisterInit(void) {
    // Настройка пинов как выходы
    DDRD |= (1 << SHIFT_DATA) | (1 << SHIFT_CLK) | (1 << SHIFT_LATCH);
    
    // Установка начального состояния
    PORTD &= ~((1 << SHIFT_DATA) | (1 << SHIFT_CLK) | (1 << SHIFT_LATCH));
    
    // Инициализация - все реле выключены
    shiftOut(0x0000);
}

void shiftOut(uint16_t data) {
    // Защелку в низкий уровень перед передачей
    PORTD &= ~(1 << SHIFT_LATCH);
    
    // Передача 16 бит (2 байта для каскадных регистров)
    for (int8_t i = 15; i >= 0; i--) {
        // Установка данных
        if (data & (1 << i)) {
            PORTD |= (1 << SHIFT_DATA);
        } else {
            PORTD &= ~(1 << SHIFT_DATA);
        }
        
        // Тактовый импульс
        PORTD |= (1 << SHIFT_CLK);
        _delay_us(1);
        PORTD &= ~(1 << SHIFT_CLK);
        _delay_us(1);
    }
    
    // Защелку в высокий уровень для обновления выходов
    PORTD |= (1 << SHIFT_LATCH);
    _delay_us(1);
    PORTD &= ~(1 << SHIFT_LATCH);
}

void updateBandSwitches(uint8_t band) {
    uint16_t shift = 0;
    
    switch(band) {
        case 0:  shift = 0b0000000000000000; break; // All - все реле выключены
        case 1:  shift = 0b1000000000000000; break; // 10m
        case 2:  shift = 0b0100000000000000; break; // 15m  
        case 3:  shift = 0b0010000000000000; break; // 20m
        case 4:  shift = 0b0001000000000000; break; // 30m
        case 5:  shift = 0b0000100000000000; break; // 40m
        case 6:  shift = 0b0000010000000000; break; // 80m
        case 7:  shift = 0b0000001000000000; break; // 160m
        default: shift = 0b0000000000000000; break;
    }
    
    shiftOut(shift);
}

// ==================== S-METER ФУНКЦИИ ====================

uint16_t readSMeterADC(void) {
    // Выбираем канал S-meter (PC1)
    ADMUX = (1 << REFS0) | (1 << MUX0); // AVCC reference, channel ADC1 (PC1)
    
    // Запускаем преобразование
    ADCSRA |= (1 << ADSC);
    
    // Ждем завершения преобразования
    while (ADCSRA & (1 << ADSC));
    
    // Возвращаем результат
    return ADC;
}

const char* getSMeterString(uint16_t adcValue) {
    // Определяем уровень S-meter по порогам
    for (int8_t i = SMETER_THRESHOLDS_COUNT - 1; i >= 0; i--) {
        if (adcValue >= smeter_thresholds[i]) {
            return smeter_strings[i];
        }
    }
    return "S0";
}

void updateSMeter(void) {
    smeterValue = readSMeterADC();
}

// ==================== ОСНОВНЫЕ ФУНКЦИИ ====================

int main(void) {
    setup();
    
    while(1) {
        processEncoder();
        handleButtons();
        
        // Обновляем S-meter каждые SMETER_UPDATE_INTERVAL циклов
        smeterUpdateCounter++;
        if (smeterUpdateCounter >= SMETER_UPDATE_INTERVAL) {
            updateSMeter();
            smeterUpdateCounter = 0;
        }
        
        updateDisplay(currentFrequency);
        _delay_ms(10);
    }
}

void setup(void) {
    // Настройка пинов для ГУНов (исправленные пины)
    DDRB |= (1 << VCO_39_48) | (1 << VCO_46_57);
    DDRC |= (1 << VCO_55_69) | (1 << VCO_68_82);
    PORTB &= ~((1 << VCO_39_48) | (1 << VCO_46_57));
    PORTC &= ~((1 << VCO_55_69) | (1 << VCO_68_82));
    
    // Настройка пинов для кнопок и S-meter на PC0 и PC1
    DDRC &= ~((1 << BUTTON_ADC_PIN) | (1 << SMETER_ADC_PIN));
    
    // Настройка пинов энкодера
    DDRD &= ~((1 << ENCODER_PIN1) | (1 << ENCODER_PIN2));
    PORTD |= (1 << ENCODER_PIN1) | (1 << ENCODER_PIN2);
    
    // Инициализация сдвиговых регистров
    shiftRegisterInit();
    
    // Настройка ADC для кнопок и S-meter
    ADMUX = (1 << REFS0); // Reference = AVCC, начальный канал ADC0
    ADCSRA = (1 << ADEN) | (1 << ADPS2) | (1 << ADPS1); // Prescaler = 64
    
    // Настройка прерываний для энкодера
    PCICR |= (1 << PCIE2);
    PCMSK2 |= (1 << PCINT18) | (1 << PCINT19);
    
    // Настройка I2C и Si5351
    i2cInit();
    _delay_ms(100);
    
    oled_init();
    _delay_ms(100);
    
    si5351a_init();  
    _delay_ms(100);
	
si5351aOutputOff(SI_CLK1_CONTROL);  // Отключить CLK1
si5351aOutputOff(SI_CLK2_CONTROL);  // Отключить CLK2
    
    
    // Установка начальной частоты
    frequency = frequency_1[currentBand];
    currentFrequency = frequency;
    si5351aSetFrequency(frequency);  // Исправлено на si5351aSetFrequency()
    updateVCO(frequency);
    updateBandSwitches(currentBand);
    
    // Инициализация состояния энкодера
    oldAB = (PIND & ((1 << ENCODER_PIN1) | (1 << ENCODER_PIN2))) >> 2;
    
    sei();
}

void updateVCO(uint32_t receiveFrequency) {
    uint32_t vcoFrequency = receiveFrequency + IF_FREQ;
    
    PORTB &= ~((1 << VCO_39_48) | (1 << VCO_46_57));
    PORTC &= ~((1 << VCO_55_69) | (1 << VCO_68_82));
    
    if (vcoFrequency >= 39000000 && vcoFrequency <= 48000000) {
        PORTB |= (1 << VCO_39_48);
    }
    else if (vcoFrequency >= 46000000 && vcoFrequency <= 57000000) {
        PORTB |= (1 << VCO_46_57);
    }
    else if (vcoFrequency >= 55000000 && vcoFrequency <= 69000000) {
        PORTC |= (1 << VCO_55_69);
    }
    else if (vcoFrequency >= 68000000 && vcoFrequency <= 82000000) {
        PORTC |= (1 << VCO_68_82);
    }
}

void changeBand(uint8_t newBand) {
    if (newBand < 8) {
        currentBand = newBand;
        frequency = frequency_1[currentBand];
        currentFrequency = frequency;
        si5351aSetFrequency(frequency);  // Исправлено на si5351aSetFrequency()
        updateVCO(frequency);
        updateBandSwitches(currentBand);
        ritOffset = 0;
    }
}

button_t readButtonADC(void) {
    // Переключаемся на канал кнопок (PC0)
    ADMUX = (1 << REFS0); // AVCC reference, channel ADC0 (PC0)
    
    ADCSRA |= (1 << ADSC);
    while (ADCSRA & (1 << ADSC));
    
    uint16_t adcValue = ADC;   
    
    if (adcValue < 114)       return BUTTON_NONE;
    else if (adcValue < 227)  return BUTTON_BAND_UP;
    else if (adcValue < 341)  return BUTTON_BAND_DN;
    else if (adcValue < 455)  return BUTTON_STEP;
    else if (adcValue < 568)  return BUTTON_RIT;
    else if (adcValue < 682)  return BUTTON_PTT;
    else return BUTTON_NONE;
}

void handleButtonPress(button_t button) {
    switch(button) {
        case BUTTON_BAND_UP:
            if (currentBand < 7) changeBand(currentBand + 1);
            break;
        case BUTTON_BAND_DN:
            if (currentBand > 0) changeBand(currentBand - 1);
            break;
        case BUTTON_STEP:
            // Новые значения шага: 5, 10, 100, 1000, 10000, 100000 Гц
            switch(stepSize) {
                case 5: stepSize = 10; break;
                case 10: stepSize = 100; break;
                case 100: stepSize = 1000; break;
                case 1000: stepSize = 10000; break;
                case 10000: stepSize = 100000; break;
                case 100000: stepSize = 5; break;
                default: stepSize = 5; break;
            }
            break;
        case BUTTON_RIT:
            ritEnabled = !ritEnabled;
            if (ritEnabled) {
                ritOffset = 0;
                currentFrequency = frequency + ritOffset;
            } else {
                currentFrequency = frequency;
                ritOffset = 0;
            }
            si5351aSetFrequency(currentFrequency);  // Исправлено на si5351aSetFrequency()
            break;
        case BUTTON_PTT:
            pttPressed = 1;
            currentFrequency = frequency;
            si5351aSetFrequency(currentFrequency);  // Исправлено на si5351aSetFrequency()
            break;
        default:
            break;
    }
}

void handleButtonRelease(button_t button) {
    switch(button) {
        case BUTTON_PTT:
            pttPressed = 0;
            if (ritEnabled) {
                currentFrequency = frequency + ritOffset;
                si5351aSetFrequency(currentFrequency);  // Исправлено на si5351aSetFrequency()
            }
            break;
        default:
            break;
    }
}

void handleButtons(void) {
    button_t currentButton = readButtonADC();
    
    for (button_t button = BUTTON_BAND_UP; button < BUTTON_COUNT; button++) {
        uint8_t isPressed = (currentButton == button);
        
        if (isPressed != lastButtonState[button]) {
            buttonDebounceCounters[button]++;
            
            if (buttonDebounceCounters[button] >= DEBOUNCE_DELAY) {
                lastButtonState[button] = isPressed;
                
                if (isPressed) {
                    handleButtonPress(button);
                } else {
                    handleButtonRelease(button);
                }
                
                buttonDebounceCounters[button] = 0;
            }
        } else {
            buttonDebounceCounters[button] = 0;
        }
    }
}

// Функция для расчета ширины строки
uint8_t get_string_width(const char *str, const FontDef *font) {
    uint8_t width = 0;
    while (*str) {
        width += font->width + 1; // символ + промежуток
        str++;
    }
    if (width > 0) width--; // убираем последний промежуток
    return width;
}

void updateDisplay(uint32_t freq) {
    // Полностью очищаем дисплей
    oled_clear();
    
    // === S-METER (правый верхний угол) ===
    const char* smeter_str = getSMeterString(smeterValue);
    uint8_t smeter_width = get_string_width(smeter_str, status_font);
    uint8_t smeter_x = OLED_WIDTH - smeter_width;
    oled_write_string(smeter_x, 0, smeter_str, status_font);
    
    // === СТРОКА СТАТУСА (левый верхний угол) - маленький шрифт ===
    char status[20];
    uint8_t status_x = 0;
    
    if (ritEnabled) {
        sprintf(status, "RIT:%+ld", ritOffset);
        // Центрируем RIT по середине, если нет S-meter или он короткий
        if (smeter_width < 20) {
            uint8_t status_width = get_string_width(status, status_font);
            status_x = (OLED_WIDTH - status_width) / 2;
        }
    } else if (pttPressed) {
        sprintf(status, "TX");
        status_x = 0; // TX в левом углу
    } else {
        sprintf(status, "RX");
        status_x = 0; // RX в левом углу
    }
    
    oled_write_string(status_x, 0, status, status_font);
    
    // === ОСНОВНАЯ ЧАСТОТА (центральная часть) - большой шрифт ===
    char buffer[20];
    
    if (freq >= 10000000) {
        // Формат XX.XXX.XXX
        sprintf(buffer, "%02lu.%03lu.%03lu", 
                freq / 1000000, 
                (freq % 1000000) / 1000, 
                freq % 1000);
    } else if (freq >= 1000000) {
        // Формат X.XXX.XXX  
        sprintf(buffer, "%lu.%03lu.%03lu", 
                freq / 1000000, 
                (freq % 1000000) / 1000, 
                freq % 1000);
    } else {
        // Формат XXX.XXX
        sprintf(buffer, "%03lu.%03lu", 
                freq / 1000, 
                freq % 1000);
    }
    
    // Центрируем и отображаем частоту
    uint8_t freq_width = get_string_width(buffer, freq_font);
    uint8_t freq_x = (OLED_WIDTH - freq_width) / 2;
    oled_write_string(freq_x, 1, buffer, freq_font);
    
    // === ИНФОРМАЦИОННАЯ СТРОКА (нижняя строка) - маленький шрифт ===
    char info_band[10];
    char info_step[20];
    
    // Формируем строку диапазона (без "B:")
    sprintf(info_band, "%s", band_names[currentBand]);
    
    // Формируем строку шага с надписью "Step" и сокращениями для kHz
    if (stepSize >= 1000) {
        // Для шагов от 1000 Гц и выше используем сокращение "k"
        uint32_t step_khz = stepSize / 1000;
        sprintf(info_step, "Step:%luk", step_khz);
    } else {
        // Для шагов меньше 1000 Гц отображаем полностью
        sprintf(info_step, "Step:%lu", stepSize);
    }
    
    // Диапазон - слева, шаг - справа
    oled_write_string(0, 3, info_band, info_font);
    
    // Вычисляем позицию для шага (правый край)
    uint8_t step_width = get_string_width(info_step, info_font);
    uint8_t step_x = OLED_WIDTH - step_width;
    oled_write_string(step_x, 3, info_step, info_font);
    
    // Обновляем дисплей
    oled_update();
}

void processEncoder(void) {
    int8_t localCount;
    
    ATOMIC_BLOCK(ATOMIC_RESTORESTATE) {
        localCount = encoderCount;
        encoderCount = 0;
    }
    
    // Ограничиваем максимальную скорость изменения
    if (localCount > 20) localCount = 20;
    if (localCount < -20) localCount = -20;
    
    if (localCount != 0) {
        if (ritEnabled && !pttPressed) {
            // Режим RIT - настраиваем смещение
            int32_t newOffset = ritOffset + (localCount * (int32_t)stepSize);
            
            // Ограничиваем диапазон RIT (±5kHz)
            if (newOffset < -RIT_MAX_OFFSET) newOffset = -RIT_MAX_OFFSET;
            if (newOffset > RIT_MAX_OFFSET) newOffset = RIT_MAX_OFFSET;
            
            ritOffset = newOffset;
            currentFrequency = frequency + ritOffset;
        } else {
            // Основное изменение частоты
            int32_t newFreq = frequency + (localCount * (int32_t)stepSize);
            
            // Проверка границ диапазона
            if (newFreq < (int32_t)freq_min[currentBand]) 
                newFreq = freq_min[currentBand];
            if (newFreq > (int32_t)freq_max[currentBand]) 
                newFreq = freq_max[currentBand];
            
            frequency = newFreq;
            currentFrequency = frequency + (ritEnabled && !pttPressed ? ritOffset : 0);
            updateVCO(frequency);
        }
        
        // Устанавливаем новую частоту
        si5351aSetFrequency(currentFrequency);  
    }
}